/******************************************************************
 *FILE: atmel_mxt_intr.c
 *SW-COMPONENT: QNX atmel touch driver
 *DESCRIPTION: This source file is to configure GPIO and interrupt
 *for atmel touch device.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#include <string.h>
#include <errno.h>

#include "atmel_mxt_ts.h"

#ifndef UNITTEST
#include <sys/neutrino.h>
#include <input/mtouch_log.h>
#include <gpio_client.h>
#else
#include "ipc_header.h"
#include "touchscreen_header.h"
#include "gpio_header.h"
#include "interrupt_header.h"
#endif

unsigned short atmel_gpio_conf(struct mxt_data *dev)
{
	uint32_t gpio_number = dev->tp_intr_gpio;
	uint32_t gpio_number_tlmm;
	uint32_t gpio_config;
	int fd;

	if (-1 == (fd = gpio_open(NULL))) {
		mtouch_error(ATMEL_DEV_NAME,"gpio_open() failed");
                error_memory("Atemel_Touch: gpio_open() failed");
		goto exit_err;
	}

	if(gpio_number  <= 0 ){
		mtouch_error(ATMEL_DEV_NAME,"Bad GPIO input param gpio=%d \n",gpio_number);
                error_memory("Atmel_Touch: Bad GPIO input param gpio=%d \n",gpio_number);
		goto exit_err;
	}

	gpio_number_tlmm = gpio_number & 0xFFFF; // for TLMM upper 16 bit mask is 0s
	gpio_config = GPIO_PIN_CFG(GPIO_INPUT,GPIO_PULL_UP,GPIO_STRENGTH_2MA,0);

	/* set gpio configuration for the gpio */
	if (GPIO_SUCCESS != gpio_set_config(fd, gpio_number_tlmm, 0, gpio_config)){
		mtouch_error(ATMEL_DEV_NAME,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
                error_memory("Atmel_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
		goto exit_err;
	}
	
	/* set interrupt cfg for the gpio */
	if (GPIO_SUCCESS != gpio_set_interrupt_cfg(fd, gpio_number_tlmm, GPIO_INTERRUPT_TRIGGER_LOW, NULL)){
		mtouch_error(ATMEL_DEV_NAME,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
                error_memory("Atmel_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
		goto exit_err;
	}
	
	/* get interrupt irq number from the configured interrupt gpio */
	if (GPIO_SUCCESS != gpio_get_interrupt_cfg(fd, gpio_number_tlmm, &dev->tp_intr)){
		mtouch_error(ATMEL_DEV_NAME," failed to get irq for gpio gpio_number %d \n", gpio_number);
                error_memory("Atmel_Touch: failed to get irq for gpio gpio_number %d \n", gpio_number);
		goto exit_err;
	}

	mtouch_info(ATMEL_DEV_NAME," irq corresponding to gpio %d is %d - exiting\n", gpio_number, dev->tp_intr);

	return 0;
exit_err:
	return -1;
}

int
atmel_tp_intr_attach(struct mxt_data *dev)
{
	int ret;
	
	ret = atmel_gpio_conf(dev);
	if (ret == -1) {
		mtouch_error(ATMEL_DEV_NAME,"faled to set gpio configuration \n ");
                error_memory("Atmel_Touch: faled to set gpio configuration \n ");
		goto fail1;
	}
	
	SIGEV_PULSE_INIT(&dev->tp_intrevent, dev->coid, MXT_PULSE_PRIORITY, MXT_PULSE_CODE, 0);

	dev->tp_iid = InterruptAttachEvent(dev->tp_intr, &dev->tp_intrevent, _NTO_INTR_FLAGS_TRK_MSK | _NTO_INTR_FLAGS_PROCESS);
	if (dev->tp_iid == -1) {
		mtouch_error(ATMEL_DEV_NAME, "InterruptAttach: %s", strerror(errno));
                error_memory("Atmel_Touch: InterruptAttach: %s", strerror(errno));
		goto fail1;
	}

	mtouch_info(ATMEL_DEV_NAME, "Attached to device with interrupt %d", dev->tp_intr);

	return 0;

fail1:
	return -1;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/atmel/atmel_tp_intr.c $ $Rev: 809524 $")
#endif
